/*
 * input.c -- GPS protocol input code.
 *
 * Copyright (c) 2000 Tero Kivinen <kivinen@iki.fi>
 */
/*
 *        Program: tgps
 *	  $Source: /u/kivinen/gps/tgps/RCS/input.h,v $
 *	  Author : $Author: kivinen $
 *
 *	  Creation          : 19:35 Aug  4 2000 kivinen
 *	  Last Modification : 18:54 Aug 23 2001 kivinen
 *	  Last check in     : $Date: 2001/08/23 15:56:13 $
 *	  Revision number   : $Revision: 1.2 $
 *	  State             : $State: Exp $
 *	  Version	    : 1.8
 *	  Edit time	    : 7 min
 *
 *	  Description       : GPS protocol input code
 *
 *	  $Log: input.h,v $
 *	  Revision 1.2  2001/08/23 15:56:13  kivinen
 *	  	Added tgps_get_hexbytes and tgps_get_hexbytes_3 prototypes.
 *
 *	  Revision 1.1  2000/08/05 00:45:09  kivinen
 *	  	Created.
 *
 *	  $EndLog$
 */

#ifndef TGPS_INPUT_H
#define TGPS_INPUT_H

#include "tgps.h"

/* Return 0 if error, and negative number if input line ended before all
   functions ware used. Return 1 if ok, return 2 if there is extra data after
   the item. */
int tgps_get_and_parse_line(Tgps conn, void *data, ...);

/* Return 0 in case of error, and 1 if ok. */
int tgps_get_and_parse_options(Tgps conn, void *data, char *ptr, ...);

/* Get position double. Return 0 in case of error, and 1 if ok. */
int tgps_get_pos_double(Tgps conn, char *ptr, double *dret);

/* Get position. Return 0 in case of error, and 1 if ok. */
int tgps_get_pos(Tgps conn, char *ptr, long *l);

/* Convert month name to number */
int tgps_month_to_number(const char *name);

/* Get time from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_get_time(Tgps conn, void *data, char *ptr,
		  void *data_ptr, size_t data_len);

/* Get float from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_get_float(Tgps conn, void *data, char *ptr,
		   void *data_ptr, size_t data_len);

/* Get 3 floats from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_get_float_3(Tgps conn, char *ptr, float *f1, float *f2, float *f3);

/* Get long integer from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_get_long(Tgps conn, void *data, char *ptr,
		  void *data_ptr, size_t data_len);

/* Get int from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_get_word(Tgps conn, void *data, char *ptr,
		  void *data_ptr, size_t data_len);

/* Get byte from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_get_byte(Tgps conn, void *data, char *ptr,
		  void *data_ptr, size_t data_len);

/* Get boolean from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_get_boolean(Tgps conn, void *data, char *ptr,
		     void *data_ptr, size_t data_len);

/* Get bytes from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_get_bytes(Tgps conn, void *data, char *ptr,
		   void *data_ptr, size_t data_len);

/* Get hexbytes from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_get_hexbytes(Tgps conn, void *data, char *ptr,
		      void *data_ptr, size_t data_len);

/* Get 3 hex bytes from the buffer. Return 0 in case of error, and 1 if ok. */
int tgps_get_hexbytes_3(Tgps conn, void *data, char *ptr,
			void *data_ptr, size_t data_len);

/* Get string from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_get_string(Tgps conn, void *data, char *ptr,
		    void *data_ptr, size_t data_len);

/* Get color from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_get_color(Tgps conn, void *data, char *ptr,
		   void *data_ptr, size_t data_len);

/* Check for w1 format. Return 0 in case of error, and 1 if ok. */
int tgps_wpt_check_for_w1(Tgps conn, TgpsWaypoint w, char *ptr);

/* Get ident string. Return 0 in case of error, and 1 if ok. */
int tgps_wpt_get_ident(Tgps conn, TgpsWaypoint w, char *ptr);

/* Get link or comment field. Return 0 in case of error, and 1 if ok. */
int tgps_wpt_get_lnk_or_comment(Tgps conn, TgpsWaypoint w, char *ptr);

/* Get latitude. Return 0 in case of error, and 1 if ok. */
int tgps_wpt_get_lat(Tgps conn, TgpsWaypoint w, char *ptr);

/* Get longitude. Return 0 in case of error, and 1 if ok. */
int tgps_wpt_get_lon(Tgps conn, TgpsWaypoint w, char *ptr);

/* Get symbol id from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_wpt_get_symbol(Tgps conn, void *data, char *ptr,
			void *data_ptr, size_t data_len);

/* Get display type from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_wpt_get_display(Tgps conn, void *data, char *ptr,
			 void *data_ptr, size_t data_len);

/* Get city from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_wpt_get_city(Tgps conn, void *data, char *ptr,
		      void *data_ptr, size_t data_len);

/* Get alt from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_wpt_get_alt(Tgps conn, void *data, char *ptr,
		     void *data_ptr, size_t data_len);

/* Get options. Return 0 in case of error, and 1 if ok. */
int tgps_wpt_get_options(Tgps conn, TgpsWaypoint w, char *ptr);

/* Get waypoint structure. Return 0 in case of error, and 1 if ok. */
int tgps_get_waypoint_v1(Tgps conn, TgpsWaypoint w);

/* Check for w format. Return 0 in case of error, and 1 if ok. */
int tgps_wpt_check_for_w(Tgps conn, TgpsWaypoint w, char *ptr);

/* Get time from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_wpt_get_time(Tgps conn, void *data, char *ptr,
		      void *data_ptr, size_t data_len);

/* Get waypoint structure. Return 0 in case of error, and 1 if ok. */
int tgps_get_waypoint_gpstrans(Tgps conn, TgpsWaypoint w);

/* Get waypoint structure. Return 0 in case of error, and 1 if ok. */
int tgps_get_waypoint(Tgps conn, TgpsWaypoint w);

/* Check for r format. Return 0 in case of error, and 1 if ok. */
int tgps_rhdr_check_for_r(Tgps conn, TgpsRouteHeader r, char *ptr);

/* Get route number. Return 0 in case of error, and 1 if ok. */
int tgps_rhdr_get_route_number(Tgps conn, TgpsRouteHeader r, char *ptr);

/* Get route header comment field. Return 0 in case of error, and 1 if ok. */
int tgps_rhdr_get_route_comment(Tgps conn, TgpsRouteHeader r, char *ptr);

/* Get route header structure in gpstrans format. Return 0 in case of error,
   and 1 if ok. */
int tgps_get_route_header_gpstrans(Tgps conn, TgpsRouteHeader r);

/* Check for r1 format. Return 0 in case of error, and 1 if ok. */
int tgps_rhdr_check_for_r1(Tgps conn, TgpsRouteHeader r, char *ptr);

/* Get route header structure. Return 0 in case of error, and 1 if ok. */
int tgps_get_route_header_v1(Tgps conn, TgpsRouteHeader r);

/* Get route header structure. Return 0 in case of error, and 1 if ok. */
int tgps_get_route_header(Tgps conn, TgpsRouteHeader r);

/* Check for k1 format. Return 0 in case of error, and 1 if ok. */
int tgps_rlnk_check_for_k1(Tgps conn, TgpsRouteLnk r, char *ptr);

/* Get route link ident string. Return 0 in case of error, and 1 if ok. */
int tgps_rlnk_get_ident(Tgps conn, TgpsRouteLnk r, char *ptr);

/* Get options. Return 0 in case of error, and 1 if ok. */
int tgps_rlnk_get_options(Tgps conn, TgpsRouteLnk r, char *ptr);

/* Get route link structure. Return 0 in case of error, and 1 if ok. */
int tgps_get_route_link(Tgps conn, TgpsRouteLnk r);

/* Check for t format. Return 0 in case of error, and 1 if ok. */
int tgps_trk_check_for_t(Tgps conn, TgpsTrack t, char *ptr);

/* Get time from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_trk_get_time(Tgps conn, void *data, char *ptr,
		      void *data_ptr, size_t data_len);

/* Get latitude. Return 0 in case of error, and 1 if ok. */
int tgps_trk_get_lat(Tgps conn, TgpsTrack t, char *ptr);

/* Get longitude. Return 0 in case of error, and 1 if ok. */
int tgps_trk_get_lon(Tgps conn, TgpsTrack t, char *ptr);

/* Get track point structure in gpstrans format. Return 0 in case of error, and
   1 if ok. */
int tgps_get_track_gpstrans(Tgps conn, TgpsTrack t);

/* Check for t1 format. Return 0 in case of error, and 1 if ok. */
int tgps_trk_check_for_t1(Tgps conn, TgpsTrack t, char *ptr);

/* Get options. Return 0 in case of error, and 1 if ok. */
int tgps_trk_get_options(Tgps conn, TgpsTrack t, char *ptr);

/* Get track point structure. Return 0 in case of error, and 1 if ok. */
int tgps_get_track_v1(Tgps conn, TgpsTrack t);

/* Get track point structure. Return 0 in case of error, and 1 if ok. */
int tgps_get_track(Tgps conn, TgpsTrack t);

/* Check for n1 format. Return 0 in case of error, and 1 if ok. */
int tgps_trk_hdr_check_for_n1(Tgps conn, TgpsTrackHeader t, char *ptr);

/* Get options. Return 0 in case of error, and 1 if ok. */
int tgps_trk_hdr_get_options(Tgps conn, TgpsTrackHeader t, char *ptr);

/* Get ident string. Return 0 in case of error, and 1 if ok. */
int tgps_trk_hdr_get_ident(Tgps conn, TgpsTrackHeader t, char *ptr);

/* Get track header structure. Return 0 in case of error, and 1 if ok. */
int tgps_get_track_hdr(Tgps conn, TgpsTrackHeader t);

/* Check for c1 format. Return 0 in case of error, and 1 if ok. */
int tgps_clock_check_for_c1(Tgps conn, TgpsTime t, char *ptr);

/* Get time from buffer. Return 0 in case of error, and 1 if ok. */
int tgps_clock_get_time(Tgps conn, void *data, char *ptr,
			void *data_ptr, size_t data_len);

/* Get time structure */
int tgps_get_clock(Tgps conn, TgpsTime t);

/* Check for p1 format. Return 0 in case of error, and 1 if ok. */
int tgps_position_check_for_p1(Tgps conn, TgpsPosition p, char *ptr);

/* Get latitude. Return 0 in case of error, and 1 if ok. */
int tgps_position_get_lat(Tgps conn, TgpsPosition p, char *ptr);

/* Get longitude. Return 0 in case of error, and 1 if ok. */
int tgps_position_get_lon(Tgps conn, TgpsPosition p, char *ptr);

/* Get position structure */
int tgps_get_position(Tgps conn, TgpsPosition p);

/* Check for l1 format. Return 0 in case of error, and 1 if ok. */
int tgps_pvt_check_for_l1(Tgps conn, TgpsPvt p, char *ptr);

/* Get latitude. Return 0 in case of error, and 1 if ok. */
int tgps_pvt_get_lat(Tgps conn, TgpsPvt p, char *ptr);

/* Get longitude. Return 0 in case of error, and 1 if ok. */
int tgps_pvt_get_lon(Tgps conn, TgpsPvt p, char *ptr);

/* Get alt. Return 0 in case of error, and 1 if ok. */
int tgps_pvt_get_alt(Tgps conn, TgpsPvt p, char *ptr);

/* Get mean seal level. Return 0 in case of error, and 1 if ok. */
int tgps_pvt_get_msl(Tgps conn, TgpsPvt p, char *ptr);

/* Get speed. Return 0 in case of error, and 1 if ok. */
int tgps_pvt_get_speed(Tgps conn, TgpsPvt p, char *ptr);

/* Get epe. Return 0 in case of error, and 1 if ok. */
int tgps_pvt_get_epe(Tgps conn, TgpsPvt p, char *ptr);

/* Get fix. Return 0 in case of error, and 1 if ok. */
int tgps_pvt_get_fix(Tgps conn, TgpsPvt p, char *ptr);

/* Get time. Return 0 in case of error and 1 if ok. */
int tgps_pvt_get_time(Tgps conn, TgpsPvt p, char *ptr);

/* Get PVT structure */
int tgps_get_pvt(Tgps conn, TgpsPvt p);

/* Almanac context */
typedef struct TgpsAlmanacContextRec *TgpsAlmanacContext,
  TgpsAlmanacContextStruct;

/* Check for a1 format. Return 0 in case of error, and 1 if ok. */
int tgps_almanac_check_for_a1(Tgps conn, TgpsAlmanacContext aa, char *ptr);

/* Get satellite id. Return 0 in case of error, and 1 if ok. */
int tgps_almanac_get_svid(Tgps conn, TgpsAlmanacContext aa, char *ptr);

/* Get data item name. Return 0 in case of error, and 1 if ok. */
int tgps_almanac_get_name(Tgps conn, TgpsAlmanacContext aa, char *ptr);

/* Get data item data. Return 0 in case of error, and 1 if ok. */
int tgps_almanac_get_data(Tgps conn, TgpsAlmanacContext aa, char *ptr);

/* Get almanac structure */
int tgps_get_almanac_v1(Tgps conn, TgpsAlmanac a);

/* Get almanac structure */
int tgps_get_almanac(Tgps conn, TgpsAlmanac a);

#endif /* TGPS_INPUT_H */
