/*
 * garmin.c -- GPS garmin tables.
 *
 * Copyright (c) 2000 Tero Kivinen <kivinen@iki.fi>
 */
/*
 *        Program: tgps
 *	  $Source: /u/kivinen/gps/tgps/RCS/garmin.c,v $
 *	  Author : $Author: kivinen $
 *
 *	  Creation          : 22:42 Apr 20 2000 kivinen
 *	  Last Modification : 20:27 Aug 18 2003 kivinen
 *	  Last check in     : $Date: 2003/08/18 20:10:21 $
 *	  Revision number   : $Revision: 1.9 $
 *	  State             : $State: Exp $
 *	  Version	    : 1.238
 *	  Edit time	    : 125 min
 *
 *	  Description       : GPS garmin tables.
 *
 *	  $Log: garmin.c,v $
 *	  Revision 1.9  2003/08/18 20:10:21  kivinen
 *	  	Added support for GPS V. Fixed packet size and actual size
 *	  	comparisions.
 *
 *	  Revision 1.8  2001/08/23 15:56:49  kivinen
 *	  	Fixed eTrex name.
 *
 *	  Revision 1.7  2001/08/23 15:40:11  kivinen
 *	  	Added eMap and eTreck Vista.
 *
 *	  Revision 1.6  2000/08/05 00:46:21  kivinen
 *	  	Removed warnings.
 *
 *	  Revision 1.5  2000/07/23 20:22:41  kivinen
 *	  	Added fix string conversion.
 *
 *	  Revision 1.4  2000/07/15 20:25:16  kivinen
 *	  	No changes.
 *
 *	  Revision 1.3  2000/07/06 23:06:31  kivinen
 *	  	Added initial upload support.
 *
 *	  Revision 1.2  2000/04/30 01:03:17  kivinen
 *	  	Updated to rev 03 document.
 *
 *	  Revision 1.1  2000/04/29 16:40:05  kivinen
 *	  	Created.
 *
 *	  $EndLog$
 */

#include "tgps.h"
#include "enum.h"

struct TgpsEnumTableRec tgps_garmin_products[] =
{
  { "GNC 250", 52 },
  { "GNC 250 XL", 64 },
  { "GNC 300", 33 },
  { "GNC 300 XL", 98 },
  { "GPS 12", 77 },
  { "GPS 12", 87 },
  { "GPS 12", 96 },
  { "GPS 12 XL", 77 },
  { "GPS 12 XL", 96 },
  { "GPS 12 XL Chinese", 106 }, 
  { "GPS 12 XL Japanese", 105 }, 
  { "GPS 120", 47 }, 
  { "GPS 120 Chinese", 55 }, 
  { "GPS 120 XL", 74 }, 
  { "GPS 125 Sounder", 61 }, 
  { "GPS 126", 95 }, 
  { "GPS 126 Chinese", 100 }, 
  { "GPS 128", 95 }, 
  { "GPS 128 Chinese", 100 }, 
  { "GPS 150", 20 }, 
  { "GPS 150 XL", 64 }, 
  { "GPS 155", 34 }, 
  { "GPS 155 XL", 98 }, 
  { "GPS 165", 34 }, 
  { "GPS 38", 41 }, 
  { "GPS 38 Chinese", 56 }, 
  { "GPS 38 Japanese", 62 }, 
  { "GPS 40", 31 }, 
  { "GPS 40", 41 }, 
  { "GPS 40 Chinese", 56 }, 
  { "GPS 40 Japanese", 62 }, 
  { "GPS 45", 31 }, 
  { "GPS 45", 41 }, 
  { "GPS 45 Chinese", 56 }, 
  { "GPS 45 XL", 41 }, 
  { "GPS 48", 96 },
  { "GPS 50", 7 },
  { "GPS 55", 14 }, 
  { "GPS 55 AVD", 15 }, 
  { "GPS 65", 18 }, 
  { "GPS 75", 13 }, 
  { "GPS 75", 23 }, 
  { "GPS 75", 42 }, 
  { "GPS 85", 25 }, 
  { "GPS 89", 39 }, 
  { "GPS 90", 45 }, 
  { "GPS 92", 112 }, 
  { "GPS 95", 24 }, 
  { "GPS 95", 35 }, 
  { "GPS 95 AVD", 22 }, 
  { "GPS 95 AVD", 36 }, 
  { "GPS 95 XL", 36 }, 
  { "GPS II", 59 }, 
  { "GPS II Plus", 73 }, 
  { "GPS II Plus", 97 }, 
  { "GPS III", 72 }, 
  { "GPS III Pilot", 71 }, 
  { "GPS III Plus", 119 }, 
  { "GPSCOM 170", 50 }, 
  { "GPSCOM 190", 53 }, 
  { "GPSMAP 130", 49 }, 
  { "GPSMAP 130 Chinese", 76 }, 
  { "GPSMAP 135 Sounder", 49 }, 
  { "GPSMAP 175", 49 }, 
  { "GPSMAP 195", 48 }, 
  { "GPSMAP 205", 29 }, 
  { "GPSMAP 205", 44 }, 
  { "GPSMAP 210", 29 }, 
  { "GPSMAP 215", 88 }, 
  { "GPSMAP 220", 29 }, 
  { "GPSMAP 225", 88 }, 
  { "GPSMAP 230", 49 },
  { "GPSMAP 230 Chinese", 76 }, 
  { "GPSMAP 235 Sounder", 49 },
  { "eMap", 111 },
  { "GPS V", 155 },
  { "eTrex Vista", 169 }
};

int tgps_garmin_products_cnt =
sizeof(tgps_garmin_products) / sizeof(tgps_garmin_products[0]);

/* Find Garmin product name based on the product id */
const char *tgps_find_garmin_name(int product_id)
{
  const char *s;

  s = tgps_number_to_string(tgps_garmin_products,
			    tgps_garmin_products_cnt,
			    product_id);
  if (s == NULL)
    return "Unknown";
  return s;
}

/* Maximum number of protocols in the tables below */
#define TGPS_MAX_PROTOCOLS 16

/* Protocol by product id table, if version is not 0, then it is the first
   version not using that entry. */
typedef struct TgpsGarminProtocolsRec {
  int product_id;
  int software_version;
  int protocols[TGPS_MAX_PROTOCOLS];
} *TgpsGarminProtocols;

/* Find the product from the protocol table */
void tgps_find_protocol_ids(int product_id, int software_version,
			    int **table, int *cnt,
			    TgpsGarminProtocols protocols,
			    int protocols_cnt);

struct TgpsGarminProtocolsRec tgps_link_protocol_ids[] =
{
  { 7, 0, { 1 } },
  { 13, 0, { 1 } },
  { 14, 0, { 1 } },
  { 15, 0, { 1 } },
  { 18, 0, { 1 } },
  { 20, 0, { 2 } },
  { 22, 0, { 1 } },
  { 23, 0, { 1 } },
  { 24, 0, { 1 } },
  { 25, 0, { 1 } },
  { 29, 0, { 1 } },
  { 31, 0, { 1 } },
  { 33, 0, { 2 } },
  { 34, 0, { 2 } },
  { 35, 0, { 1 } },
  { 36, 0, { 1 } },
  { 39, 0, { 1 } },
  { 41, 0, { 1 } },
  { 42, 0, { 1 } },
  { 44, 0, { 1 } },
  { 45, 0, { 1 } },
  { 47, 0, { 1 } },
  { 48, 0, { 1 } },
  { 49, 0, { 1 } },
  { 50, 0, { 1 } },
  { 52, 0, { 2 } },
  { 53, 0, { 1 } },
  { 55, 0, { 1 } },
  { 56, 0, { 1 } },
  { 59, 0, { 1 } },
  { 61, 0, { 1 } },
  { 62, 0, { 1 } },
  { 64, 0, { 2 } },
  { 71, 0, { 1 } },
  { 72, 0, { 1 } },
  { 73, 0, { 1 } },
  { 74, 0, { 1 } },
  { 76, 0, { 1 } },
  { 77, 0, { 1 } },
  { 87, 0, { 1 } },
  { 88, 0, { 1 } },
  { 95, 0, { 1 } },
  { 96, 0, { 1 } },
  { 97, 0, { 1 } },
  { 98, 0, { 2 } },
  { 100, 0, { 1 } },
  { 105, 0, { 1 } },
  { 106, 0, { 1 } },
  { 111, 0, { 1 } },
  { 112, 0, { 1 } },
  { 119, 0, { 1 } },
  { 155, 0, { 1 } },
  { 169, 0, { 1 } }
};

int tgps_link_protocol_id_cnt =
sizeof(tgps_link_protocol_ids) / sizeof(tgps_link_protocol_ids[0]);

struct TgpsGarminProtocolsRec tgps_appl_protocol_ids[] =
{
  {   7,   0, { 10, 100, 200,           500, 600, 700 } },
  {  13,   0, { 10, 100, 200, 300, 400, 500, 600, 700 } },
  {  14,   0, { 10, 100, 200,      400, 500, 600, 700 } },
  {  15,   0, { 10, 100, 200,      400, 500, 600, 700 } },
  {  18,   0, { 10, 100, 200, 300, 400, 500, 600, 700 } },
  {  20,   0, { 11, 100, 200,      400, 500, 600, 700 } },
  {  22,   0, { 10, 100, 200, 300, 400, 500, 600, 700 } },
  {  23,   0, { 10, 100, 200, 300, 400, 500, 600, 700 } },
  {  24,   0, { 10, 100, 200, 300, 400, 500, 600, 700 } },
  {  25,   0, { 10, 100, 200, 300, 400, 500, 600, 700 } },
  {  29,   0, { 10, 100, 200, 300, 400, 500, 600, 700 } },
  {  31,   0, { 10, 100, 200, 300,      500, 600, 700 } },
  {  33,   0, { 11, 100, 200,      400, 500, 600, 700 } },
  {  34,   0, { 11, 100, 200,      400, 500, 600, 700 } },
  {  35,   0, { 10, 100, 200, 300, 400, 500, 600, 700 } },
  {  36, 300, { 10, 100, 200, 300, 400, 500, 600, 700 } },
  {  36,   0, { 10, 100, 200, 300,      500, 600, 700 } },
  {  39,   0, { 10, 100, 200, 300,      500, 600, 700 } },
  {  41,   0, { 10, 100, 200, 300,      500, 600, 700 } },
  {  42,   0, { 10, 100, 200, 300, 400, 500, 600, 700 } },
  {  44,   0, { 10, 100, 200, 300, 400, 500, 600, 700 } },
  {  45,   0, { 10, 100, 200, 300,      500, 600, 700 } },
  {  47,   0, { 10, 100, 200, 300,      500, 600, 700 } },
  {  48,   0, { 10, 100, 200, 300,      500, 600, 700 } },
  {  49,   0, { 10, 100, 200, 300, 400, 500, 600, 700 } },
  {  50,   0, { 10, 100, 200, 300,      500, 600, 700 } },
  {  52,   0, { 11, 100, 200,      400, 500, 600, 700 } },
  {  53,   0, { 10, 100, 200, 300,      500, 600, 700 } },
  {  55,   0, { 10, 100, 200, 300,      500, 600, 700 } },
  {  56,   0, { 10, 100, 200, 300,      500, 600, 700 } },
  {  59,   0, { 10, 100, 200, 300,      500, 600, 700 } },
  {  61,   0, { 10, 100, 200, 300,      500, 600, 700 } },
  {  62,   0, { 10, 100, 200, 300,      500, 600, 700 } },
  {  64,   0, { 11, 100, 200,      400, 500, 600, 700 } },
  {  71,   0, { 10, 100, 200, 300,      500, 600, 700 } },
  {  72,   0, { 10, 100, 200, 300,      500, 600, 700 } },
  {  73,   0, { 10, 100, 200, 300,      500, 600, 700 } },
  {  74,   0, { 10, 100, 200, 300,      500, 600, 700 } },
  {  76,   0, { 10, 100, 200, 300, 400, 500, 600, 700 } },
  {  77, 350, { 10, 100, 200, 300, 400, 500, 600, 700 } },
  {  77, 361, { 10, 100, 200, 300,      500, 600, 700 } },
  {  77,   0, { 10, 100, 200, 300, 400, 500, 600, 700 } },
  {  87,   0, { 10, 100, 200, 300, 400, 500, 600, 700 } },
  {  88,   0, { 10, 100, 200, 300, 400, 500, 600, 700 } },
  {  95,   0, { 10, 100, 200, 300, 400, 500, 600, 700 } },
  {  96,   0, { 10, 100, 200, 300, 400, 500, 600, 700 } },
  {  97,   0, { 10, 100, 200, 300,      500, 600, 700 } },
  {  98,   0, { 11, 100, 200,      400, 500, 600, 700 } },
  { 100,   0, { 10, 100, 200, 300, 400, 500, 600, 700 } },
  { 105,   0, { 10, 100, 200, 300, 400, 500, 600, 700 } },
  { 106,   0, { 10, 100, 200, 300, 400, 500, 600, 700 } },
  { 111,   0, { 10, 100, 200, 300,      500, 600, 700 } },
  { 112,   0, { 10, 100, 200, 300,      500, 600, 700 } },
  { 119,   0, { 10, 100, 200, 300,      500, 600, 700 } },
  { 155,   0, { 10, 100, 201, 301, 400, 500, 600, 700, 800, 900, 902, 903, 904 } },
  { 169,   0, { 10, 100, 200, 300,      500, 600, 700 } }
};

int tgps_appl_protocol_id_cnt =
sizeof(tgps_appl_protocol_ids) / sizeof(tgps_appl_protocol_ids[0]);

struct TgpsGarminProtocolsRec tgps_data_protocol_ids[] =
{
  {   7,   0, { 100, 200,           500, 600, 700 } },
  {  13,   0, { 100, 200, 300, 400, 500, 600, 700 } },
  {  14,   0, { 100, 200,      400, 500, 600, 700 } },
  {  15,   0, { 151, 200,           500, 600, 700 } },
  {  18,   0, { 100, 200, 300, 400, 500, 600, 700 } },
  {  20,   0, { 150, 201,      450, 550, 600, 700 } },
  {  22,   0, { 152, 200, 300,      500, 600, 700 } },
  {  23,   0, { 100, 200, 300, 400, 500, 600, 700 } },
  {  24,   0, { 100, 200, 300, 400, 500, 600, 700 } },
  {  25,   0, { 100, 200, 300, 400, 500, 600, 700 } },
  {  29, 400, { 101, 201, 300, 	    500, 600, 700 } },
  {  29,   0, { 102, 201, 300, 	    500, 600, 700 } },
  {  31,   0, { 100, 201, 300, 	    500, 600, 700 } },
  {  33,   0, { 150, 201,      450, 550, 600, 700 } },
  {  34,   0, { 150, 201,      450, 550, 600, 700 } },
  {  35,   0, { 100, 200, 300, 400, 500, 600, 700 } },
  {  36,   0, { 152, 200, 300, 	    500, 600, 700 } },
  {  39,   0, { 151, 201, 300, 	    500, 600, 700 } },
  {  41,   0, { 100, 201, 300, 	    500, 600, 700 } },
  {  42,   0, { 100, 200, 300, 400, 500, 600, 700 } },
  {  44,   0, { 101, 201, 300, 	    500, 600, 700 } },
  {  45,   0, { 152, 201, 300, 	    500, 600, 700 } },
  {  47,   0, { 100, 201, 300, 	    500, 600, 700 } },
  {  48,   0, { 154, 201, 300, 	    501, 600, 700 } },
  {  49,   0, { 102, 201, 300, 	    501, 600, 700 } },
  {  50,   0, { 152, 201, 300, 	    501, 600, 700 } },
  {  52,   0, { 150, 201,      450, 550, 600, 700 } },
  {  53,   0, { 152, 201, 300, 	    501, 600, 700 } },
  {  55,   0, { 100, 201, 300, 	    500, 600, 700 } },
  {  56,   0, { 100, 201, 300, 	    500, 600, 700 } },
  {  59,   0, { 100, 201, 300, 	    500, 600, 700 } },
  {  61,   0, { 100, 201, 300, 	    500, 600, 700 } },
  {  62,   0, { 100, 201, 300, 	    500, 600, 700 } },
  {  64,   0, { 150, 201,      450, 551, 600, 700 } },
  {  71,   0, { 155, 201, 300, 	    501, 600, 700 } },
  {  72,   0, { 104, 201, 300, 	    501, 600, 700 } },
  {  73,   0, { 103, 201, 300, 	    501, 600, 700 } },
  {  74,   0, { 100, 201, 300, 	    500, 600, 700 } },
  {  76,   0, { 102, 201, 300,      501, 600, 700 } },
  {  77, 301, { 100, 201, 300, 400, 501, 600, 700 } },
  {  77, 350, { 103, 201, 300, 403, 501, 600, 700 } },
  {  77, 361, { 103, 201, 300,      501, 600, 700 } },
  {  77,   0, { 103, 201, 300, 403, 501, 600, 700 } },
  {  87,   0, { 103, 201, 300, 403, 501, 600, 700 } },
  {  88,   0, { 102, 201, 300,      501, 600, 700 } },
  {  95,   0, { 103, 201, 300, 403, 501, 600, 700 } },
  {  96,   0, { 103, 201, 300, 403, 501, 600, 700 } },
  {  97,   0, { 103, 201, 300,      501, 600, 700 } },
  {  98,   0, { 150, 201,      450, 551, 600, 700 } },
  { 100,   0, { 103, 201, 300, 403, 501, 600, 700 } },
  { 105,   0, { 103, 201, 300, 403, 501, 600, 700 } },
  { 106,   0, { 103, 201, 300, 403, 501, 600, 700 } },
  { 112,   0, { 152, 201, 300,      501, 600, 700 } },
  { 111,   0, { 108, 210, 301,      501, 600, 700 } },
  { 119,   0, { 104, 201, 300, 	    501, 600, 700 } },
  { 155,   0, { 109, 202, 210, 310, 301, 501, 600, 700, 800 } },
  { 169,   0, { 108, 202, 210, 309, 310, 509, 600, 700 } }
};

int tgps_data_protocol_id_cnt =
sizeof(tgps_data_protocol_ids) / sizeof(tgps_data_protocol_ids[0]);

/* Find the product from the protocol table */
void tgps_find_protocol_ids(int product_id, int software_version,
			    int **table, int *cnt,
			    TgpsGarminProtocols protocols,
			    int protocols_cnt)
{
  int i, j;

  free(*table);
  *table = NULL;
  *cnt = 0;

  for(i = 0; i < protocols_cnt; i++)
    {
      if (protocols[i].product_id == product_id)
	{
	  if (protocols[i].software_version == 0 ||
	      protocols[i].software_version < software_version)
	    {
	      for(j = 0; j < TGPS_MAX_PROTOCOLS; j++)
		{
		  if (protocols[i].protocols[j] == 0)
		    break;
		}
	      *cnt = j;
	      *table = calloc(j, sizeof(int));
	      if (table == NULL)
		{
		  *cnt = 0;
		  return;
		}
	      for(j = 0; j < *cnt; j++)
		(*table)[j] = protocols[i].protocols[j];
	      return;
	    }
	}
    }
  return;
}

/* Find physical protocol ids. */
void tgps_find_phys_protocol_ids(int product_id, int software_version,
				 int **table, int *cnt)
{
  free(*table);
  *cnt = 0;
  *table = NULL;
}

/* Find link protocol ids. */
void tgps_find_link_protocol_ids(int product_id, int software_version,
				 int **table, int *cnt)
{
  tgps_find_protocol_ids(product_id, software_version,
			 table, cnt, tgps_link_protocol_ids,
			 tgps_link_protocol_id_cnt);
}

/* Find application protocol ids. */
void tgps_find_appl_protocol_ids(int product_id, int software_version,
				 int **table, int *cnt)
{
  tgps_find_protocol_ids(product_id, software_version,
			 table, cnt, tgps_appl_protocol_ids,
			 tgps_appl_protocol_id_cnt);
}

/* Find data protocol ids. */
void tgps_find_data_protocol_ids(int product_id, int software_version,
				 int **table, int *cnt)
{
  tgps_find_protocol_ids(product_id, software_version,
			 table, cnt, tgps_data_protocol_ids,
			 tgps_data_protocol_id_cnt);
}

struct TgpsEnumTableRec tgps_garmin_symbols[] =
{
  /*-- Symbols for marine (group 0...0-8191...bits 15-13=000). --*/
  { "anchor", 0 }, /* white anchor symbol */
  { "bell", 1 }, /* white bell symbol */
  { "diamond_grn", 2 }, /* green diamond symbol */
  { "diamond_red", 3 }, /* red diamond symbol */
  { "dive1", 4 }, /* diver down flag 1 */
  { "dive2", 5 }, /* diver down flag 2 */
  { "dollar", 6 }, /* white dollar symbol */
  { "fish", 7 }, /* white fish symbol */
  { "fuel", 8 }, /* white fuel symbol */
  { "horn", 9 }, /* white horn symbol */
  { "house", 10 }, /* white house symbol */
  { "knife", 11 }, /* white knife & fork symbol */
  { "light", 12 }, /* white light symbol */
  { "mug", 13 }, /* white mug symbol */
  { "skull", 14 }, /* white skull and crossbones symbol*/
  { "square_grn", 15 }, /* green square symbol */
  { "square_red", 16 }, /* red square symbol */
  { "wbuoy", 17 }, /* white buoy waypoint symbol */
  { "wpt_dot", 18 }, /* waypoint dot */
  { "wreck", 19 }, /* white wreck symbol */
  { "null", 20 }, /* null symbol (transparent) */
  { "mob", 21 }, /* man overboard symbol */

  /*-- marine navaid symbols --*/
  { "buoy_ambr", 22 }, /* amber map buoy symbol */
  { "buoy_blck", 23 }, /* black map buoy symbol */
  { "buoy_blue", 24 }, /* blue map buoy symbol */
  { "buoy_grn", 25 }, /* green map buoy symbol */
  { "buoy_grn_red", 26 }, /* green/red map buoy symbol */
  { "buoy_grn_wht", 27 }, /* green/white map buoy symbol */
  { "buoy_orng", 28 }, /* orange map buoy symbol */
  { "buoy_red", 29 }, /* red map buoy symbol */
  { "buoy_red_grn", 30 }, /* red/green map buoy symbol */
  { "buoy_red_wht", 31 }, /* red/white map buoy symbol */
  { "buoy_violet", 32 }, /* violet map buoy symbol */
  { "buoy_wht", 33 }, /* white map buoy symbol */
  { "buoy_wht_grn", 34 }, /* white/green map buoy symbol */
  { "buoy_wht_red", 35 }, /* white/red map buoy symbol */
  { "dot", 36 }, /* white dot symbol */
  { "rbcn", 37 }, /* radio beacon symbol */

  /*-- leave space for more navaids (up to 128 total) --*/
  { "boat_ramp", 150 }, /* boat ramp symbol */
  { "boat", 150 }, /* Conversion from garmin 16 set */
  { "camp", 151 }, /* campground symbol */
  { "restrooms", 152 }, /* restrooms symbol */
  { "showers", 153 }, /* shower symbol */
  { "drinking_wtr", 154 }, /* drinking water symbol */
  { "phone", 155 }, /* telephone symbol */
  { "1st_aid", 156 }, /* first aid symbol */
  { "info", 157 }, /* information symbol */
  { "parking", 158 }, /* parking symbol */
  { "park", 159 }, /* park symbol */
  { "picnic", 160 }, /* picnic symbol */
  { "scenic", 161 }, /* scenic area symbol */
  { "skiing", 162 }, /* skiing symbol */
  { "swimming", 163 }, /* swimming symbol */
  { "dam", 164 }, /* dam symbol */
  { "controlled", 165 }, /* controlled area symbol */
  { "danger", 166 }, /* danger symbol */
  { "restricted", 167 }, /* restricted area symbol */
  { "null_2", 168 }, /* null symbol */
  { "ball", 169 }, /* ball symbol */
  { "car", 170 }, /* car symbol */
  { "deer", 171 }, /* deer symbol */
  { "shpng_cart", 172 }, /* shopping cart symbol */
  { "lodging", 173 }, /* lodging symbol */
  { "mine", 174 }, /* mine symbol */
  { "trail_head", 175 }, /* trail head symbol */
  { "truck_stop", 176 }, /* truck stop symbol */
  { "user_exit", 177 }, /* user exit symbol */
  { "exit", 177 }, /* Conversion from garmin 16 set */
  { "flag", 178 }, /* flag symbol */
  { "circle_x", 179 }, /* circle with x in the center */
    
  /*-- Symbols for land (group 1...8192-16383...bits 15-13=001). --*/
  { "is_hwy", 8192 }, /* interstate hwy symbol */
  { "us_hwy", 8193 }, /* us hwy symbol */
  { "st_hwy", 8194 }, /* state hwy symbol */
  { "mi_mrkr", 8195 }, /* mile marker symbol */
  { "trcbck", 8196 }, /* TracBack (feet) symbol */
  { "back_track", 8196 }, /* Conversion from garmin 16 set */
  { "golf", 8197 }, /* golf symbol */
  { "sml_cty", 8198 }, /* small city symbol */
  { "med_cty", 8199 }, /* medium city symbol */
  { "lrg_cty", 8200 }, /* large city symbol */
  { "freeway", 8201 }, /* intl freeway hwy symbol */
  { "ntl_hwy", 8202 }, /* intl national hwy symbol */
  { "cap_cty", 8203 }, /* capitol city symbol (star) */
  { "amuse_pk", 8204 }, /* amusement park symbol */
  { "bowling", 8205 }, /* bowling symbol */
  { "car_rental", 8206 }, /* car rental symbol */
  { "car_repair", 8207 }, /* car repair symbol */
  { "fastfood", 8208 }, /* fast food symbol */
  { "fitness", 8209 }, /* fitness symbol */
  { "movie", 8210 }, /* movie symbol */
  { "museum", 8211 }, /* museum symbol */
  { "pharmacy", 8212 }, /* pharmacy symbol */
  { "pizza", 8213 }, /* pizza symbol */
  { "post_ofc", 8214 }, /* post office symbol */
  { "rv_park", 8215 }, /* RV park symbol */
  { "school", 8216 }, /* school symbol */
  { "stadium", 8217 }, /* stadium symbol */
  { "store", 8218 }, /* dept. store symbol */
  { "zoo", 8219 }, /* zoo symbol */
  { "gas_plus", 8220 }, /* convenience store symbol */
  { "gas", 8220 }, /* Conversion from garmin 16 set */
  { "faces", 8221 }, /* live theater symbol */
  { "ramp_int", 8222 }, /* ramp intersection symbol */
  { "st_int", 8223 }, /* street intersection symbol */
  { "weigh_sttn", 8226 }, /* inspection/weigh station symbol */
  { "toll_booth", 8227 }, /* toll booth symbol */
  { "elev_pt", 8228 }, /* elevation point symbol */
  { "ex_no_srvc", 8229 }, /* exit without services symbol */
  { "geo_place_mm", 8230 }, /* Geographic place name, man-made */
  { "geo_place_wtr", 8231 }, /* Geographic place name, water */
  { "geo_place_lnd", 8232 }, /* Geographic place name, land */
  { "bridge", 8233 }, /* bridge symbol */
  { "building", 8234 }, /* building symbol */
  { "cemetery", 8235 }, /* cemetery symbol */
  { "church", 8236 }, /* church symbol */
  { "civil", 8237 }, /* civil location symbol */
  { "crossing", 8238 }, /* crossing symbol */
  { "hist_town", 8239 }, /* historical town symbol */
  { "levee", 8240 }, /* levee symbol */
  { "military", 8241 }, /* military location symbol */
  { "oil_field", 8242 }, /* oil field symbol */
  { "tunnel", 8243 }, /* tunnel symbol */
  { "beach", 8244 }, /* beach symbol */
  { "forest", 8245 }, /* forest symbol */
  { "summit", 8246 }, /* summit symbol */
  { "lrg_ramp_int", 8247 }, /* large ramp intersection symbol */
  { "lrg_ex_no_srvc", 8248 }, /* large exit without services smbl */
  { "badge", 8249 }, /* police/official badge symbol */
  { "cards", 8250 }, /* gambling/casino symbol */
  { "snowski", 8251 }, /* snow skiing symbol */
  { "iceskate", 8252 }, /* ice skating symbol */
  { "wrecker", 8253 }, /* tow truck (wrecker) symbol */
  { "border", 8254 }, /* border crossing (port of entry) */

  /*-- Symbols for aviation (group 2...16383-24575...bits 15-13=010). --*/
  { "airport", 16384 }, /* airport symbol */
  { "int", 16385 }, /* intersection symbol */
  { "ndb", 16386 }, /* non-directional beacon symbol */
  { "vor", 16387 }, /* VHF omni-range symbol */
  { "heliport", 16388 }, /* heliport symbol */
  { "private", 16389 }, /* private field symbol */
  { "soft_fld", 16390 }, /* soft field symbol */
  { "tall_tower", 16391 }, /* tall tower symbol */
  { "short_tower", 16392 }, /* short tower symbol */
  { "glider", 16393 }, /* glider symbol */
  { "ultralight", 16394 }, /* ultralight symbol */
  { "parachute", 16395 }, /* parachute symbol */
  { "vortac", 16396 }, /* VOR/TACAN symbol */
  { "vordme", 16397 }, /* VOR-DME symbol */
  { "faf", 16398 }, /* first approach fix */
  { "lom", 16399 }, /* localizer outer marker */
  { "map", 16400 }, /* missed approach point */
  { "tacan", 16401 }, /* TACAN symbol */
  { "seaplane", 16402 } /* Seaplane Base */
};

int tgps_garmin_symbols_cnt =
sizeof(tgps_garmin_symbols) / sizeof(tgps_garmin_symbols[0]);

struct TgpsEnumTableRec tgps_garmin_symbols_16[] =
{
  { "dot", 0 }, /* dot symbol */
  { "house", 1 }, /* house symbol */
  { "gas", 2 }, /* gas symbol */
  { "car", 3 }, /* car symbol */
  { "fish", 4 }, /* fish symbol */
  { "boat", 5 }, /* boat symbol */
  { "anchor", 6 }, /* anchor symbol */
  { "wreck", 7 }, /* wreck symbol */
  { "exit", 8 }, /* exit symbol */
  { "skull", 9 }, /* skull symbol */
  { "flag", 10 }, /* flag symbol */
  { "camp", 11 }, /* camp symbol */
  { "circle_x", 12 }, /* circle with x symbol */
  { "deer", 13 }, /* deer symbol */
  { "1st_aid", 14 }, /* first aid symbol */
  { "back_track", 15 } /* back track symbol */
};

int tgps_garmin_symbols_16_cnt =
sizeof(tgps_garmin_symbols_16) / sizeof(tgps_garmin_symbols_16[0]);

/* Convert symbol number to string */
const char *tgps_symbol_to_string(TgpsGarminSymbolType type,
				  int symbol)
{
  const char *s = NULL;

  switch (type)
    {
    case TGPS_GARMIN_SYMBOLS_FULL:
      s = tgps_number_to_string(tgps_garmin_symbols, tgps_garmin_symbols_cnt,
				symbol);
      break;
    case TGPS_GARMIN_SYMBOLS_16:
      s = tgps_number_to_string(tgps_garmin_symbols_16,
				tgps_garmin_symbols_16_cnt,
				symbol);
      break;
    }
  if (s == NULL)
    return "unknown";
  return s;
}

/* Convert symbol string to number */
int tgps_string_to_symbol(TgpsGarminSymbolType type,
			  const char *name)
{
  int s;

  switch (type)
    {
    case TGPS_GARMIN_SYMBOLS_FULL:
      s = tgps_string_to_number(tgps_garmin_symbols, tgps_garmin_symbols_cnt,
				name);
      break;
    case TGPS_GARMIN_SYMBOLS_16:
      s = tgps_string_to_number(tgps_garmin_symbols_16,
				tgps_garmin_symbols_16_cnt,
				name);
      break;
    }
  return s;
}

struct TgpsEnumTableRec tgps_garmin_color[] =
{
  { "Black", 16 },
  { "Dark_Red", 17 },
  { "Dark_Green", 18 },
  { "Dark_Yellow", 19 },
  { "Dark_Blue", 20 },
  { "Dark_Magenta", 21 },
  { "Dark_Cyan", 22 },
  { "Light_Gray", 23 },
  { "Dark_Gray", 24 },
  { "Red", 25 },
  { "Green", 26 },
  { "Yellow", 27 },
  { "Blue", 28 },
  { "Magenta", 29 },
  { "Cyan", 30 },
  { "White", 31 },
  { "Red", 1 },
  { "Green", 2 },
  { "Blue", 3 },
};

int tgps_garmin_color_cnt =
sizeof(tgps_garmin_color) / sizeof(tgps_garmin_color[0]);

/* Convert color number to string */
const char *tgps_color_to_string(int color)
{
  const char *s = NULL;

  s = tgps_number_to_string(tgps_garmin_color,
			    tgps_garmin_color_cnt,
			    color);
  if (s == NULL)
    return "unknown";
  return s;
}

/* Convert color string to number */
int tgps_string_to_color(const char *name)
{
  int s;

  s = tgps_string_to_number(tgps_garmin_color,
			    tgps_garmin_color_cnt,
			    name);
  if (s == -1)
    return 0;
  return s;
}

struct TgpsEnumTableRec tgps_garmin_fix[] =
{
  { "unusable", 0 },
  { "invalid", 1 },
  { "2D", 2 },
  { "3D", 3 },
  { "2D_diff", 4 },
  { "3D_diff", 5 }
};

int tgps_garmin_fix_cnt =
sizeof(tgps_garmin_fix) / sizeof(tgps_garmin_fix[0]);

/* Convert fix number to string */
const char *tgps_fix_to_string(int fix)
{
  const char *s = NULL;

  s = tgps_number_to_string(tgps_garmin_fix,
			    tgps_garmin_fix_cnt,
			    fix);
  if (s == NULL)
    return "unknown";
  return s;
}

/* Convert fix string to number */
int tgps_string_to_fix(const char *name)
{
  return tgps_string_to_number(tgps_garmin_fix,
			       tgps_garmin_fix_cnt,
			       name);
}
